<?php

namespace LivemeshAddons;

// Exit if accessed directly
use Analog\Utils;
use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Group_Control_Image_Size;

if (!defined('ABSPATH'))
    exit;


if (!class_exists('LELA_Image_Generator')) :


    /**
     * Main LELA_Image_Generator Class
     *
     */
    class LELA_Image_Generator {


        /**
         * LELA OpenAI Client Object
         *
         * @var object|LELA_OpenAI_Client
         */
        public $openai_client;

        /**
         * Get things going
         *
         * @since 1.4.4
         */
        public function __construct() {

            add_action( 'elementor/element/image/section_image/before_section_end', array( $this, 'register_image_generation_controls' ), 20, 2 );

            $this->openai_client = new LELA_OpenAI_Client();

            add_action('wp_ajax_nopriv_lela_generate_image', array($this, 'generate_image'));

            add_action('wp_ajax_lela_generate_image', array($this, 'generate_image'));

        }

        /**
         * Register Image Generation Controls. Make sure you prefix everything to avoid conflict with Elementor widget
         *
         * @param Controls_Stack $element Elementor element.
         * @param string         $section_id Section ID.
         */
        public function register_image_generation_controls( Controls_Stack $element, $section_id ) {

            $element->add_control(
                'lela_image_generation',
                [
                    'label'     => esc_html__('AI Image Generation', 'livemesh-el-assistant'),
                    'type'      => Controls_Manager::HEADING,
                    'separator' => 'before'
                ]
            );

            $element->add_control(
                'lela_prompt',
                [
                    'type' => Controls_Manager::TEXTAREA,
                    'label' => esc_html__('Prompt for Image Generation', 'livemesh-el-assistant'),
                    'label_block' => true,
                    'rows' => 2,
                    'description' => esc_html__('Specify instructions for image generation.', 'livemesh-el-assistant'),
                    'dynamic' => [
                        'active' => true,
                    ],
                ]
            );

            $element->add_control(
                'lela_image_style',
                [
                    'type' => Controls_Manager::SELECT,
                    'label' => esc_html__('Generated Image Style', 'livemesh-el-assistant'),
                    'description' => esc_html__('Specify the style for the image generated.', 'livemesh-el-assistant'),
                    'options' => array(
                        'None' => esc_html__('- Do Not Specify -', 'livemesh-el-assistant'),
                        '3d' => esc_html__('3D', 'livemesh-el-assistant'),
                        '60s Flat Illustration' => esc_html__('60s Flat Illustration', 'livemesh-el-assistant'),
                        'Abstract' => esc_html__('Abstract', 'livemesh-el-assistant'),
                        'Modern' => esc_html__('Modern', 'livemesh-el-assistant'),
                        'Cartoon' => esc_html__('Cartoon', 'livemesh-el-assistant'),
                        'Comic Book' => esc_html__('Comic Book', 'livemesh-el-assistant'),
                        'Cyberpunk' => esc_html__('Cyberpunk', 'livemesh-el-assistant'),
                        'Fantasy' => esc_html__('Fantasy', 'livemesh-el-assistant'),
                        'Futurism' => esc_html__('Futurism', 'livemesh-el-assistant'),
                        'Noir' => esc_html__('Noir', 'livemesh-el-assistant'),
                        'Oil Painting' => esc_html__('Oil Painting', 'livemesh-el-assistant'),
                        'Pencil Sketch' => esc_html__('Pencil Sketch', 'livemesh-el-assistant'),
                        'Photorealism' => esc_html__('Photorealism', 'livemesh-el-assistant'),
                        'Stained Glass' => esc_html__('Stained Glass', 'livemesh-el-assistant'),
                        'Synthwave' => esc_html__('Synthwave', 'livemesh-el-assistant'),
                        'Graffiti' => esc_html__('Graffiti', 'livemesh-el-assistant'),
                        'Contemporary' => esc_html__('Contemporary', 'livemesh-el-assistant'),
                        'Surrealism' => esc_html__('Surrealism', 'livemesh-el-assistant'),
                        'Popart' => esc_html__('Popart', 'livemesh-el-assistant'),
                        'Impressionist' => esc_html__('Impressionist', 'livemesh-el-assistant'),
                        'Cubism' => esc_html__('Cubism', 'livemesh-el-assistant'),
                        'Watercolor' => esc_html__('Watercolor', 'livemesh-el-assistant'),
                    ),
                    'default' => 'None',
                ]
            );

            $element->add_control(
                'lela_image_size',
                [
                    'type' => Controls_Manager::SELECT,
                    'label' => esc_html__('Generated Image Size', 'livemesh-el-assistant'),
                    'description' => esc_html__('Specify the size for the image to be generated. Post generation, you can specify preferred custom image size in the Image Size option above.', 'livemesh-el-assistant'),
                    'options' => array(
                        '256x256' => esc_html__('Small (256x256)', 'livemesh-el-assistant'),
                        '512x512' => esc_html__('Medium (512x512)', 'livemesh-el-assistant'),
                        '1024x1024' => esc_html__('Big (1024x1024)', 'livemesh-el-assistant'),
                    ),
                    'default' => '512x512',
                ]
            );

            $element->add_control(
                'lela_generate',
                [
                    'type' => Controls_Manager::BUTTON,
                    'label' => '',
                    'separator' => 'before',
                    'show_label' => false,
                    'text' => esc_html__('Generate', 'livemesh-el-assistant'),
                    'button_type' => 'default',
                    'event' => 'lela:image:generate'

                ]
            );

        }

        /**
         * @throws \Exception
         */
        public function generate_image() {

            check_ajax_referer('lela-assistant', '_ajax_nonce-lela-assistant');

            if (!current_user_can('manage_options')) {
                wp_send_json_error( esc_html__('Error! Your account does not have user permission to generate content!', 'livemesh-el-assistant'));
            }

            $image_params = array();

            if (isset($_POST['imageParams'])) {
                $image_params = array_map('sanitize_text_field', $_POST['imageParams']);
            }

            try {

                if (empty($image_params['prompt'])) {
                    throw new \Exception(esc_html__('You have not provided the instruction for the image that you want to generate!', 'livemesh-el-assistant'));
                }

                // For now, let the prompt be the title for the image
                $image_alt = $image_params['prompt'];
                $prompt = 'Generate an image of ' . $image_params['prompt'] . '\n';
                if ($image_params['imageStyle'] != 'None') {
                    $prompt .= 'Image Style: ' . $image_params['imageStyle'] . '\n';
                }

                $url = 'https://api.openai.com/v1/images/generations';

                $call_params = array(
                    'prompt' => $prompt,
                    'n' => 1,
                    'size' => $image_params['imageSize'],
                    'response_format' => 'url',
                );

                $response = $this->openai_client->dispatch($url, $call_params);

                $image_url = isset($response['data'][0]['url']) ? $response['data'][0]['url'] : '';

                if (!function_exists('media_sideload_image')) {
                    require_once(ABSPATH . 'wp-admin/includes/media.php');
                    require_once(ABSPATH . 'wp-admin/includes/file.php');
                    require_once(ABSPATH . 'wp-admin/includes/image.php');
                }
                // Save to the media library and return me image html
                $image_id = media_sideload_image($image_url, 0, null, 'id');
                if (is_wp_error($image_id)) {
                    throw new \Exception($image_id->get_error_message());
                }
                $image_src = wp_get_attachment_url($image_id);

                $success = [
                    'message' => esc_html__('Image generated successfully.', 'livemesh-el-assistant'),
                    'image' => array(
                        'id' => $image_id,
                        'url' => $image_src,
                        'alt' => $image_alt,
                        'source' => 'library'
                    )
                ];
                wp_send_json_success($success);

            } catch (\Throwable $throwable) {
                wp_send_json_error(esc_html__('Error! ', 'livemesh-el-assistant') . $throwable->getMessage());
            }
        }

    }

endif; // End if class_exists check


