<?php

/*
Widget Name: Advanced Writer
Description: Generates content using advanced AI tools
Author: LiveMesh
Author URI: https://www.livemeshwp.com
*/

namespace LivemeshAddons\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Scheme_Color;
use Elementor\Group_Control_Typography;
use Elementor\Scheme_Typography;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Class for Advanced Writer widget that generates text content for the web page
 */
class LELA_Advanced_Writer_Widget extends Widget_Base {

    /**
     * Get the name for the widget
     * @return string
     */
    public function get_name() {
        return 'lela-advanced-writer';
    }

    /**
     * Get the widget title
     * @return string|void
     */
    public function get_title() {
        return esc_html__('AI Advanced Writer', 'livemesh-el-assistant');
    }

    /**
     * Get the widget icon
     * @return string
     */
    public function get_icon() {
        return 'eicon-text-area';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * @return string[]
     */
    public function get_categories() {
        return array('livemesh-addons');
    }

    /**
     * Get the widget documentation URL
     * @return string
     */
    public function get_custom_help_url() {
        return 'https://livemeshelementor.com/docs/livemesh-addons/core-addons/ai-advanced-writer/';
    }

    /**
     * Obtain the scripts required for the widget to function
     * @return string[]
     */
    public function get_script_depends() {
        return [];
    }

    /**
     * Register the controls for the widget
     * Adds fields that help configure and customize the widget
     * @return void
     */
    protected function register_controls() {

        $this->start_controls_section(
            'section_content',
            [
                'label' => esc_html__('Advanced AI Content', 'livemesh-el-assistant'),
            ]
        );

        $this->add_control(
            'content',
            [
                'type' => Controls_Manager::WYSIWYG,
                'label_block' => true,
                'separator' => 'before',
                'default' => esc_html__('AI generated content reflected here', 'livemesh-el-assistant'),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_control(
            'prompt',
            [
                'type' => Controls_Manager::TEXTAREA,
                'label' => esc_html__('Prompt', 'livemesh-el-assistant'),
                'label_block' => true,
                'placeholder' => '',
                'rows' => 10,
                'description' => wp_kses_post(__('Provide subject and detailed instructions for the content you want to generate. You can also paste here existing content and request AI to fix style/grammar or rewrite that content for you. <br><strong><div style="font-size: 14px; font-style: normal; margin-top: 10px; color: red;" >This widget is deprecated in favor of native Text Editor widget of Elementor which is now AI enabled with better features. <br></br>Hence, the Generate feature of AI Advanced Writer widget is now disabled. <u>This widget will be removed in future releases. To preserve existing content, pls ensure all data is moved to Text Editor widget.</u></div></strong>', 'livemesh-el-assistant')),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_control(
            'generate',
            [
                'type' => Controls_Manager::BUTTON,
                'label' => '',
                'separator' => 'before',
                'show_label' => false,
                'text' => esc_html__('Generate', 'livemesh-el-assistant'),
                'button_type' => 'default',

            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_content_params',
            [
                'label' => esc_html__('Content Parameters', 'livemesh-el-assistant'),
                'tab' => Controls_Manager::TAB_SETTINGS,
            ]
        );

        $this->add_control(
            'keywords',
            [
                'type' => Controls_Manager::TEXT,
                'label' => esc_html__('Keywords', 'livemesh-el-assistant'),
                'label_block' => true,
                'description' => esc_html__('Provide keywords separated by commas for SEO', 'livemesh-el-assistant'),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_control(
            'language',
            [
                'type' => Controls_Manager::SELECT,
                'label' => esc_html__('Choose Language', 'livemesh-el-assistant'),
                'description' => esc_html__('The text will be generated in the selected language.', 'livemesh-el-assistant'),
                'options' => array(
                    'English' => esc_html__('English', 'livemesh-el-assistant'),
                    'Spanish' => esc_html__('Spanish', 'livemesh-el-assistant'),
                    'German' => esc_html__('German', 'livemesh-el-assistant'),
                    'French' => esc_html__('French', 'livemesh-el-assistant'),
                    'Italian' => esc_html__('Italian', 'livemesh-el-assistant'),
                    'Portuguese' => esc_html__('Portuguese', 'livemesh-el-assistant'),
                    'Romanian' => esc_html__('Romanian', 'livemesh-el-assistant'),
                    'Dutch' => esc_html__('Dutch', 'livemesh-el-assistant'),
                    'Polish' => esc_html__('Polish', 'livemesh-el-assistant'),
                    'Finnish' => esc_html__('Finnish', 'livemesh-el-assistant'),
                    'Danish' => esc_html__('Danish', 'livemesh-el-assistant'),
                    'Norwegian' => esc_html__('Norwegian', 'livemesh-el-assistant'),
                    'Chinese' => esc_html__('Chinese', 'livemesh-el-assistant'),
                    'Hindi' => esc_html__('Hindi', 'livemesh-el-assistant'),
                    'Japanese' => esc_html__('Japanese', 'livemesh-el-assistant'),
                    'Korean' => esc_html__('Korean', 'livemesh-el-assistant'),
                    'Indonesian' => esc_html__('Indonesian', 'livemesh-el-assistant'),
                    'Russian' => esc_html__('Russian', 'livemesh-el-assistant'),
                    'Arabic' => esc_html__('Arabic', 'livemesh-el-assistant'),
                    'Turkish' => esc_html__('Turkish', 'livemesh-el-assistant'),
                    'Ukrainian' => esc_html__('Ukrainian', 'livemesh-el-assistant'),
                    'Swedish' => esc_html__('Swedish', 'livemesh-el-assistant'),
                    'Hungarian' => esc_html__('Hungarian', 'livemesh-el-assistant'),
                    'Greek' => esc_html__('Greek', 'livemesh-el-assistant'),
                ),
                'default' => 'English',
            ]
        );

        $this->add_control(
            'writing_style',
            [
                'type' => Controls_Manager::SELECT,
                'label' => esc_html__('Writing Style', 'livemesh-el-assistant'),
                'description' => esc_html__('Specify the appropriate writing style.', 'livemesh-el-assistant'),
                'options' => array(
                    'Informative' => esc_html__('Informative', 'livemesh-el-assistant'),
                    'Descriptive' => esc_html__('Descriptive', 'livemesh-el-assistant'),
                    'Creative' => esc_html__('Creative', 'livemesh-el-assistant'),
                    'Blog' => esc_html__('Blog', 'livemesh-el-assistant'),
                    'News' => esc_html__('News', 'livemesh-el-assistant'),
                    'Casual' => esc_html__('Casual', 'livemesh-el-assistant'),
                    'Technical' => esc_html__('Technical', 'livemesh-el-assistant'),
                    'Concise' => esc_html__('Concise', 'livemesh-el-assistant'),
                    'Narrative' => esc_html__('Narrative', 'livemesh-el-assistant'),
                    'Reflective' => esc_html__('Reflective', 'livemesh-el-assistant'),
                    'Academic' => esc_html__('Academic', 'livemesh-el-assistant'),
                    'Analytical' => esc_html__('Analytical', 'livemesh-el-assistant'),
                    'Evaluative' => esc_html__('Evaluative', 'livemesh-el-assistant'),
                    'Journalistic' => esc_html__('Journalistic', 'livemesh-el-assistant'),
                    'Argumentative' => esc_html__('Argumentative', 'livemesh-el-assistant'),
                    'Dramatic' => esc_html__('Dramatic', 'livemesh-el-assistant'),
                    'Dialogue' => esc_html__('Dialogue', 'livemesh-el-assistant'),
                    'Satirical' => esc_html__('Satirical', 'livemesh-el-assistant'),
                    'Critical' => esc_html__('Critical', 'livemesh-el-assistant'),
                    'Persuasive' => esc_html__('Persuasive', 'livemesh-el-assistant'),
                    'Detailed' => esc_html__('Detailed', 'livemesh-el-assistant'),
                    'Monologue' => esc_html__('Monologue', 'livemesh-el-assistant'),
                    'Letter' => esc_html__('Letter', 'livemesh-el-assistant'),
                    'Lyrical' => esc_html__('Lyrical', 'livemesh-el-assistant'),
                    'Poetic' => esc_html__('Poetic', 'livemesh-el-assistant'),
                    'Articulate' => esc_html__('Articulate', 'livemesh-el-assistant'),
                    'Anecdotal' => esc_html__('Anecdotal', 'livemesh-el-assistant'),
                    'Personal' => esc_html__('Personal', 'livemesh-el-assistant'),
                    'Comparative' => esc_html__('Comparative', 'livemesh-el-assistant'),
                    'Fiction' => esc_html__('Fiction', 'livemesh-el-assistant'),
                    'Pastoral' => esc_html__('Pastoral', 'livemesh-el-assistant'),
                    'Sensory' => esc_html__('Sensory', 'livemesh-el-assistant'),
                    'Simple' => esc_html__('Simple', 'livemesh-el-assistant'),
                    'Vivid' => esc_html__('Vivid', 'livemesh-el-assistant'),
                    'Historical' => esc_html__('Historical', 'livemesh-el-assistant'),
                    'Biographical' => esc_html__('Biographical', 'livemesh-el-assistant'),
                    'Colloquial' => esc_html__('Colloquial', 'livemesh-el-assistant'),
                    'Expository' => esc_html__('Expository', 'livemesh-el-assistant'),
                ),
                'default' => 'Informative',
            ]
        );

        $this->add_control(
            'writing_tone',
            [
                'type' => Controls_Manager::SELECT,
                'label' => esc_html__('Writing Tone', 'livemesh-el-assistant'),
                'description' => esc_html__('Specify the tone for the generated content.', 'livemesh-el-assistant'),
                'options' => array(
                    'Neutral' => esc_html__('Neutral', 'livemesh-el-assistant'),
                    'Formal' => esc_html__('Formal', 'livemesh-el-assistant'),
                    'Informal' => esc_html__('Informal', 'livemesh-el-assistant'),
                    'Conversational' => esc_html__('Conversational', 'livemesh-el-assistant'),
                    'Scientific' => esc_html__('Scientific', 'livemesh-el-assistant'),
                    'Assertive' => esc_html__('Assertive', 'livemesh-el-assistant'),
                    'Cheerful' => esc_html__('Cheerful', 'livemesh-el-assistant'),
                    'Inspirational' => esc_html__('Inspirational', 'livemesh-el-assistant'),
                    'Journalistic' => esc_html__('Journalistic', 'livemesh-el-assistant'),
                    'Emotional' => esc_html__('Emotional', 'livemesh-el-assistant'),
                    'Narrative' => esc_html__('Narrative', 'livemesh-el-assistant'),
                    'Sarcastic' => esc_html__('Sarcastic', 'livemesh-el-assistant'),
                    'Factual' => esc_html__('Factual', 'livemesh-el-assistant'),
                    'Humorous' => esc_html__('Humorous', 'livemesh-el-assistant'),
                    'Friendly' => esc_html__('Friendly', 'livemesh-el-assistant'),
                    'Professional' => esc_html__('Professional', 'livemesh-el-assistant'),
                    'Condescending' => esc_html__('Condescending', 'livemesh-el-assistant'),
                    'Persuasive' => esc_html__('Persuasive', 'livemesh-el-assistant'),
                    'Polite' => esc_html__('Polite', 'livemesh-el-assistant'),
                    'Supportive' => esc_html__('Supportive', 'livemesh-el-assistant'),
                    'Sensitive' => esc_html__('Sensitive', 'livemesh-el-assistant'),
                    'Sincere' => esc_html__('Sincere', 'livemesh-el-assistant'),
                    'Skeptical' => esc_html__('Skeptical', 'livemesh-el-assistant'),
                ),
                'default' => 'Neutral',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_image_params',
            [
                'label' => esc_html__('Image Generation', 'livemesh-el-assistant'),
                'tab' => Controls_Manager::TAB_SETTINGS,
            ]
        );

        $this->add_control(
            'image_prompt',
            [
                'type' => Controls_Manager::TEXT,
                'label' => esc_html__('Prompt for Image', 'livemesh-el-assistant'),
                'label_block' => true,
                'description' => esc_html__('Specify instructions for image generation. Leave blank if you do not want image to be generated', 'livemesh-el-assistant'),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_control(
            'image_style',
            [
                'type' => Controls_Manager::SELECT,
                'label' => esc_html__('Choose Image Style', 'livemesh-el-assistant'),
                'description' => esc_html__('Specify the style for the image generated.', 'livemesh-el-assistant'),
                'options' => array(
                        'None' => esc_html__('- Do Not Specify -', 'livemesh-el-assistant'),
                        '3d' => esc_html__('3D', 'livemesh-el-assistant'),
                        '60s Flat Illustration' => esc_html__('60s Flat Illustration', 'livemesh-el-assistant'),
                        'Abstract' => esc_html__('Abstract', 'livemesh-el-assistant'),
                        'Modern' => esc_html__('Modern', 'livemesh-el-assistant'),
                        'Cartoon' => esc_html__('Cartoon', 'livemesh-el-assistant'),
                        'Comic Book' => esc_html__('Comic Book', 'livemesh-el-assistant'),
                        'Cyberpunk' => esc_html__('Cyberpunk', 'livemesh-el-assistant'),
                        'Fantasy' => esc_html__('Fantasy', 'livemesh-el-assistant'),
                        'Futurism' => esc_html__('Futurism', 'livemesh-el-assistant'),
                        'Noir' => esc_html__('Noir', 'livemesh-el-assistant'),
                        'Oil Painting' => esc_html__('Oil Painting', 'livemesh-el-assistant'),
                        'Pencil Sketch' => esc_html__('Pencil Sketch', 'livemesh-el-assistant'),
                        'Photorealism' => esc_html__('Photorealism', 'livemesh-el-assistant'),
                        'Stained Glass' => esc_html__('Stained Glass', 'livemesh-el-assistant'),
                        'Synthwave' => esc_html__('Synthwave', 'livemesh-el-assistant'),
                        'Graffiti' => esc_html__('Graffiti', 'livemesh-el-assistant'),
                        'Contemporary' => esc_html__('Contemporary', 'livemesh-el-assistant'),
                        'Surrealism' => esc_html__('Surrealism', 'livemesh-el-assistant'),
                        'Popart' => esc_html__('Popart', 'livemesh-el-assistant'),
                        'Impressionist' => esc_html__('Impressionist', 'livemesh-el-assistant'),
                        'Cubism' => esc_html__('Cubism', 'livemesh-el-assistant'),
                        'Watercolor' => esc_html__('Watercolor', 'livemesh-el-assistant'),
                ),
                'default' => 'None',
            ]
        );

        $this->add_control(
            'image_size',
            [
                'type' => Controls_Manager::SELECT,
                'label' => esc_html__('Choose Image Size', 'livemesh-el-assistant'),
                'description' => esc_html__('Specify the size for the image to be generated.', 'livemesh-el-assistant'),
                'options' => array(
                    '256x256' => esc_html__('Small (256x256)', 'livemesh-el-assistant'),
                    '512x512' => esc_html__('Medium (512x512)', 'livemesh-el-assistant'),
                    '1024x1024' => esc_html__('Big (1024x1024)', 'livemesh-el-assistant'),
                ),
                'default' => '512x512',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_openai_api_settings',
            [
                'label' => esc_html__('OpenAI API Settings', 'livemesh-el-assistant'),
                'tab' => Controls_Manager::TAB_SETTINGS,
            ]
        );

        $this->add_control(
            'gpt3_model',
            [
                'type' => Controls_Manager::SELECT,
                'label' => esc_html__('Choose a GPT-3 Model to use', 'livemesh-el-assistant'),
                'description' => wp_kses_post(__('Know more about these models and their capabilities <a target="_blank" href="https://platform.openai.com/docs/models/overview">here</a>', 'livemesh-el-assistant')),
                'options' => array(
                    'text-davinci-003' => esc_html__('text-davinci-003', 'livemesh-el-assistant'),
                    'text-curie-001' => esc_html__('text-curie-001', 'livemesh-el-assistant'),
                    'text-babbage-001' => esc_html__('text-babbage-001', 'livemesh-el-assistant'),
                    'text-ada-001' => esc_html__('text-ada-001', 'livemesh-el-assistant'),
                ),
                'default' => 'text-davinci-003',
            ]
        );

        $this->add_control(
            'max_tokens',
            [
                'label' => esc_html__('Maximum Tokens', 'livemesh-el-assistant'),
                'description' => esc_html__('The maximum number of tokens to use for generation. Higher values means more content will be generated.', 'livemesh-el-assistant'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 1000,
                ],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 2048,
                    ],
                ],
            ]
        );

        $this->add_control(
            'temperature',
            [
                'label' => esc_html__('Temperature', 'livemesh-el-assistant'),
                'description' => esc_html__('The sampling temperature to use. Higher values means the model will take more risks. Try 0.9 for more creative applications, and 0 for ones with a well-defined answer.', 'livemesh-el-assistant'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 0.6,
                ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'step' => 0.1,
                        'max' => 1,
                    ],
                ],
            ]
        );

        $this->add_control(
            'presence_penalty',
            [
                'label' => esc_html__('Presence Penalty', 'livemesh-el-assistant'),
                'description' => esc_html__('Number between -2.0 and 2.0. Default is 0. Positive values penalize new tokens based on whether they appear in the text so far, increasing the model\'s likelihood to talk about new topics.', 'livemesh-el-assistant'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 0,
                ],
                'range' => [
                    'px' => [
                        'min' => -2.0,
                        'step' => 0.1,
                        'max' => 2.0,
                    ],
                ],
            ]
        );

        $this->add_control(
            'frequency_penalty',
            [
                'label' => esc_html__('Frequency Penalty', 'livemesh-el-assistant'),
                'description' => esc_html__('Number between -2.0 and 2.0. Default is 0. Positive values penalize new tokens based on their existing frequency in the text so far, decreasing the model\'s likelihood to repeat the same line verbatim.', 'livemesh-el-assistant'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 0,
                ],
                'range' => [
                    'px' => [
                        'min' => -2.0,
                        'step' => 0.1,
                        'max' => 2.0,
                    ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_styling',
            [
                'label' => esc_html__('Text', 'livemesh-el-assistant'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'content_color',
            [
                'label' => esc_html__('Text Color', 'livemesh-el-assistant'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .lela-content' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'content_typography',
                'label' => esc_html__('Typography', 'livemesh-el-assistant'),
                'selector' => '{{WRAPPER}} .lela-content',
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render HTML widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @return void
     */
    protected function render() {

        $settings = $this->get_settings_for_display();

        $settings = apply_filters('lela_advanced_content_' . $this->get_id() . '_settings', $settings);
        ?>
        <div class="lela-content">

            <?php echo wp_kses_post($settings['content']); ?>

        </div><!-- .lela-content -->

        <?php
    }

    /**
     * Render the widget output in the editor.
     * @return void
     */
    protected function content_template() {
    }

}