<?php
/**
 * Tutorlms_Course_Import_Export_WP_List
 */
class Tutorlms_Certificate_Verify_And_Share_WP_List extends WP_List_Table {

    /**
     * constructor.
     */
    public function __construct() {
        // Set parent defaults.
        parent::__construct( array(
            'singular' => 'tutorlms-course-certificate',     // Singular name of the listed records.
            'plural'   => 'tutorlms-course-certificate',    // Plural name of the listed records.
            'ajax'     => false,                // Does this table support ajax?
        ) );
    }

    /**
     * Get a list of columns. The format is:
     * 'internal-name' => 'Title'
     *
     * REQUIRED! This method dictates the table's columns and titles. This should
     * return an array where the key is the column slug (and class) and the value
     * is the column's title text. If you need a checkbox for bulk actions, refer
     * to the $columns array below.
     *
     * The 'cb' column is treated differently than the rest. If including a checkbox
     * column in your table you must create a `column_cb()` method. If you don't need
     * bulk actions or checkboxes, simply leave the 'cb' entry out of your array.
     *
     * @see WP_List_Table::::single_row_columns()
     * @return array An associative array containing column information.
     */
    public function get_columns() {
        $columns = array(
            'title'             => esc_html_x( 'Course Title', 'Column label', 'tutorlms-certificate-verify-and-share' ),
            'username'          => esc_html_x( 'Username', 'Column label', 'tutorlms-certificate-verify-and-share' ),
            'email'             => esc_html_x( 'User Email', 'Column label', 'tutorlms-certificate-verify-and-share' ),
            'issue_date'        => esc_html_x( 'Issue Date', 'Column label', 'tutorlms-certificate-verify-and-share' ),
            'expire_date'       => esc_html_x( 'Expire Date', 'Column label', 'tutorlms-certificate-verify-and-share' ),
            'status'            => esc_html_x( 'Status', 'Column label', 'tutorlms-certificate-verify-and-share' ),
            'action'            => esc_html_x( 'Action', 'Column label', 'tutorlms-certificate-verify-and-share' ),
        );

        return $columns;
    }

   /**
     * This method merely defines which columns should be sortable and makes them
     * clickable - it does not handle the actual sorting. You still need to detect
     * the ORDERBY and ORDER querystring variables within `prepare_items()` and sort
     * your data accordingly (usually by modifying your query).
     *
     * @return array An associative array containing all the columns that should be sortable.
     */
    protected function get_sortable_columns() {

        return $sortable_columns;
    }

    /**
     * Get default column value.
     *
     * For more detailed insight into how columns are handled, take a look at
     * WP_List_Table::single_row_columns()
     *
     * @param object $item        A singular item (one full row's worth of data).
     * @param string $column_name The name/slug of the column to be processed.
     * @return string Text or HTML to be placed inside the column <td>.
     */
    protected function column_default( $item, $column_name ) {
    }

    protected function get_certificate_username() {
        $data = $this->course_certificate_data();
    }

    /**
     * Get completed course data
     *
     * @since v.1.0.0
     */
    private function course_certificate_data() {
        global $wpdb;
        $data = $wpdb->get_results(
            "SELECT comment_ID as certificate_id,
                    comment_post_ID as course_id,
                    comment_author as completed_user_id,
                    comment_date as completion_date,
                    comment_content as completed_hash
            FROM    $wpdb->comments
            WHERE   comment_agent = 'TutorLMSPlugin'
                    AND comment_type = 'course_completed'"
        );

        return ! empty( $data ) ? $data : [];
    }
    
    protected function get_export_options() {
        
        return '<a href="' . esc_url( $url ) . '" class="tutorlms-btn">' . __( 'Export', 'tutorlms-certificate-verify-and-share' ) . '</a>';
    }

    /**
     * Get course title.
     *
     * @param object $item A singular item (one full row's worth of data).
     * @return string Text to be placed inside the column <td>.
     */
    protected function column_title( $item ) {
        return get_the_title( $item['course_id'] );
    }

    /**
     * Get username.
     *
     * @param object $item A singular item (one full row's worth of data).
     * @return string Text to be placed inside the column <td>.
     */
    protected function column_username( $item ) {

        $user_data  = get_userdata( $item[ 'completed_user_id' ] );
        $username   = $user_data->user_login;
        return $username;
    }

    /**
     * Get user email address.
     *
     * @param object $item A singular item (one full row's worth of data).
     * @return string Text to be placed inside the column <td>.
     */
    protected function column_email( $item ) {

        $user_data  = get_userdata( $item[ 'completed_user_id' ] );
        $email      = $user_data->user_email;
        return $email;
    }

    /**
     * Get certificate issue date.
     *
     * @param object $item A singular item (one full row's worth of data).
     * @return string Text to be placed inside the column <td>.
     */
    protected function column_issue_date( $item ) {
        return date( 'jS F Y', strtotime( $item['completion_date'] ) );
    }

    /**
     * Get certificate expire date.
     *
     * @param object $item A singular item (one full row's worth of data).
     * @return string Text to be placed inside the column <td>.
     */
    protected function column_expire_date( $item ) {
        $enable         = get_post_meta( $item[ 'course_id' ], '_tutor_enable_certificate_expiration', true );
        $duration       = get_post_meta( $item[ 'course_id' ], '_tutor_certificate_expiration_duration', true );
        $duration_type  = get_post_meta( $item[ 'course_id' ], '_tutor_certificate_expiration_duration_type', true );

        if( 'on' === $enable ) {
            return date( 'jS F Y', strtotime( '+'. $duration .' '. $duration_type, strtotime( $item[ 'completion_date' ] ) ) );
        } else {
            return 'Disabled for this course';
        }
    }

    /**
     * Get certificate status.
     *
     * @param object $item A singular item (one full row's worth of data).
     * @return string Text to be placed inside the column <td>.
     */
    protected function column_status( $item ) {
        $enable         = get_post_meta( $item[ 'course_id' ], '_tutor_enable_certificate_expiration', true );
        $duration       = get_post_meta( $item[ 'course_id' ], '_tutor_certificate_expiration_duration', true );
        $duration_type  = get_post_meta( $item[ 'course_id' ], '_tutor_certificate_expiration_duration_type', true );
        $expire_date    = strtotime( '+'. $duration .' '. $duration_type, strtotime( $item[ 'completion_date' ] ) );
        $current_date   = strtotime( 'now' );
        
        $forced_status  = get_comment_meta( $item['certificate_id'], '_certificate_status_'.$item['course_id'], true );

        if( 'on' !== $enable ) {
            return esc_html( 'Disable for this course', 'tutorlms-certificate-verify-and-share' );
        }
        
        if( 'on' === $enable && $current_date < $expire_date && 'expire' !== $forced_status ) {
            return esc_html_x( 'Active', 'tutorlms-certificate-verify-and-share' );
        } else {
            return esc_html_x( 'Expired', 'tutorlms-certificate-verify-and-share' );
        }
    }

    /**
     * Get action button.
     *
     * @param object $item A singular item (one full row's worth of data).
     * @return string Text to be placed inside the column <td>.
     */
    protected function column_action( $item ) {
        $enable         = get_post_meta( $item[ 'course_id' ], '_tutor_enable_certificate_expiration', true );
        $duration       = get_post_meta( $item[ 'course_id' ], '_tutor_certificate_expiration_duration', true );
        $duration_type  = get_post_meta( $item[ 'course_id' ], '_tutor_certificate_expiration_duration_type', true );
        $expire_date    = strtotime( '+'. $duration .' '. $duration_type, strtotime( $item[ 'completion_date' ] ) );
        $current_date   = strtotime( 'now' );

        $forced_status  = get_comment_meta( $item['certificate_id'], '_certificate_status_'.$item['course_id'], true );

        if( 'on' !== $enable ) {
            return esc_html( 'Disable for this course', 'tutorlms-certificate-verify-and-share' );
        }

        if( 'on' === $enable && $current_date < $expire_date && 'expire' !== $forced_status ) {
            
            $url = add_query_arg( 
                array( 
                    'page' => 'tutorlms-certificate-verify-and-share', 
                    'tab'=> 'logs', 
                    'action' => 'revoke', 
                    'course_id' => $item['course_id'], 
                    'certificate_id' => $item['certificate_id'], 
                    'user_id' => $item['completed_user_id'], 
                ), 
                admin_url('admin.php') 
            );
            
            return '<a href="'. esc_url( $url ) .'" class="tutorlms-btn">' . __( 'Revoke', 'tutorlms-certificate-verify-and-share' ) . '</a>';
        } else {
            
            $url = add_query_arg( 
                array( 
                    'page' => 'tutorlms-certificate-verify-and-share', 
                    'tab'=> 'logs', 
                    'action' => 'upgrade', 
                    'course_id' => $item['course_id'], 
                    'certificate_id' => $item['certificate_id'], 
                    'user_id' => $item['completed_user_id'], 
                ), 
                admin_url('admin.php') 
            );

            return '<a href="'. esc_url( $url ) .'" class="tutorlms-btn">' . __( 'Upgrade', 'tutorlms-certificate-verify-and-share' ) . '</a>';
        }
    }

    /**
     * Get an associative array ( option_name => option_title ) with the list
     * @return array An associative array containing all the bulk actions.
     */
    protected function get_bulk_actions() {
        // return $actions;
    }

    /**
     * Prepares the list of items for displaying.
     *
     * @global wpdb $wpdb
     * @uses $this->_column_headers
     * @uses $this->items
     * @uses $this->get_columns()
     * @uses $this->get_sortable_columns()
     * @uses $this->get_pagenum()
     * @uses $this->set_pagination_args()
     */
    function prepare_items() {
        /*
         * First, lets decide how many records per page to show
         */
        $per_page = 5;

        
        $columns  = $this->get_columns();
        $hidden   = array();
        // $sortable = $this->get_sortable_columns();

        
        $this->_column_headers = array( $columns, $hidden );
        

        $data = $this->get_tutorlms_courses();
        
        // usort( $data, array( $this, 'usort_reorder' ) );

        $current_page = $this->get_pagenum();

        $total_items = count( $data );

        $data = array_slice( $data, ( ( $current_page - 1 ) * $per_page ), $per_page );

        $this->items = $data;
        // $this->process_bulk_action();

        /**
         * REQUIRED. We also have to register our pagination options & calculations.
         */
        $this->set_pagination_args( array(
            'total_items' => $total_items,                     // WE have to calculate the total number of items.
            'per_page'    => $per_page,                        // WE have to determine how many items to show on a page.
            'total_pages' => ceil( $total_items / $per_page ), // WE have to calculate the total number of pages.
        ) );
    }

    protected function get_tutorlms_courses() {
        global $wpdb;
        $data = $wpdb->get_results(
            "SELECT comment_ID as certificate_id,
                    comment_post_ID as course_id,
                    comment_author as completed_user_id,
                    comment_date as completion_date,
                    comment_content as completed_hash
            FROM    $wpdb->comments
            WHERE   comment_agent = 'TutorLMSPlugin'
                    AND comment_type = 'course_completed'"
        , ARRAY_A );

        return ! empty( $data ) ? $data : [];
    }

    /**
     * Callback to allow sorting of example data.
     *
     * @param string $a First value.
     * @param string $b Second value.
     *
     * @return int
     */
    protected function usort_reorder( $a, $b ) {
        // If no sort, default to title.
        $orderby = ! empty( $_REQUEST['orderby'] ) ? wp_unslash( $_REQUEST['orderby'] ) : 'post_title'; // WPCS: Input var ok.

        // If no order, default to asc.
        $order = ! empty( $_REQUEST['order'] ) ? wp_unslash( $_REQUEST['order'] ) : 'asc'; // WPCS: Input var ok.

        // Determine sort order.
        $result = strcmp( $a[ $orderby ], $b[ $orderby ] );

        return ( 'asc' === $order ) ? $result : - $result;
    }
}

// Create an instance of our package class.
$Tutorlms_Certificate_Verify_And_Share_WP_List = new Tutorlms_Certificate_Verify_And_Share_WP_List();

// Fetch, prepare, sort, and filter our data.
$Tutorlms_Certificate_Verify_And_Share_WP_List->prepare_items();

?>
<div class="wn_wrap certiffc-lg-wrap">
    <!-- Forms are NOT created automatically, so you need to wrap the table in one to use features like bulk actions -->
    <form id="tutorlms-course-filter" method="get">
        <!-- For plugins, we also need to ensure that the form posts back to our current page -->
        <input type="hidden" name="page" value="tutorlms-certificate-verify-and-share" />
        <input type="hidden" name="tab" value="logs" />
        <!-- Now we can render the completed list table -->
        <?php $Tutorlms_Certificate_Verify_And_Share_WP_List->display() ?>
    </form>

</div>
