<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://wooninjas.com/
 * @since      1.0.0
 *
 * @package    Tutorlms_Certificate_Verify_And_Share
 * @subpackage Tutorlms_Certificate_Verify_And_Share/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Tutorlms_Certificate_Verify_And_Share
 * @subpackage Tutorlms_Certificate_Verify_And_Share/public
 * @author     WooNinjas <info@wooninjas.com>
 */
class Tutorlms_Certificate_Verify_And_Share_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/tutorlms-certificate-verify-and-share-public.css', array(), $this->version, 'all' );

		wp_enqueue_style( $this->plugin_name.'-fontawesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.3/css/all.min.css', array(), $this->version, 'all' );

		wp_enqueue_style( $this->plugin_name.'-modal', 'https://cdnjs.cloudflare.com/ajax/libs/jquery-modal/0.9.1/jquery.modal.min.css', array(), $this->version, 'all' );
	
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/tutorlms-certificate-verify-and-share-public.js', array( 'jquery' ), $this->version, false );

		wp_enqueue_script( $this->plugin_name.'-modal', 'https://cdnjs.cloudflare.com/ajax/libs/jquery-modal/0.9.1/jquery.modal.min.js', array( 'jquery' ), $this->version, false );

		wp_localize_script( $this->plugin_name, 'TLCVSVars', $this->get_localize_data() );

		$this->add_expiry_date_on_certificate_page();
	}

	public function get_localize_data() {
		$data = array(
			'ajaxurl'           => esc_url( admin_url( 'admin-ajax.php' ) ),
			'debug'             => defined( 'WP_DEBUG' ) ? true : false,
			'siteURL'           => site_url(),
			'_ajax_nonce'      	=> wp_create_nonce( 'TutorLMS_Share_Certificate' ),
		);

		return $data;
	}

	private function certificate_url() {
		return urlencode( get_home_url() . '?cert_hash=' . sanitize_text_field( $_GET['cert_hash'] ) );
	}

	/**
	 * Load social buttons template.
	 *
	 * @since    1.0.0
	 */
	public function load_social_buttons_template() {

		$certificate_data 	= $this->course_certificate_data( sanitize_text_field( $_GET['cert_hash'] ) );
		$issue_date 		= strtotime( $certificate_data[0]->completion_date );
		$expire_date		= $this->check_certificate_validity();		
		$current_date		= strtotime( 'now' );		
		$expiration_enable	= get_post_meta( $certificate_data[0]->course_id, '_tutor_enable_certificate_expiration', true );
		
		$forced_status  = get_comment_meta( $certificate_data[0]->certificate_id, '_certificate_status_'.$certificate_data[0]->course_id, true );

		if( empty( $forced_status ) ) {
			update_comment_meta( $certificate_data[0]->certificate_id, '_certificate_status_' . $certificate_data[0]->course_id, 'active' );
		} 

		
		if( 'on' !== $expiration_enable ) {
			if ( file_exists( plugin_dir_path( dirname( __FILE__ ) ) . 'public/partials/social-buttons.php' ) ) {
				$file = plugin_dir_path( dirname( __FILE__ ) ) . 'public/partials/social-buttons.php';
				require_once $file;
			}

		} else {
			if( $current_date < $expire_date && 'expire' !== $forced_status ) {
				if ( file_exists( plugin_dir_path( dirname( __FILE__ ) ) . 'public/partials/social-buttons.php' ) ) {
					$file = plugin_dir_path( dirname( __FILE__ ) ) . 'public/partials/social-buttons.php';
					require_once $file;
				}
			} else {
				$content = '<div class="certificate-expire">
	        					<p>Your certificate is expired, please contact site administrator.</p>
	    					</div>';
	    		echo $content;			
				exit();
			}
		}

		
	}

	/**
	 * Check certificate validity
	 *
	 * @since v.1.0.0
	 */
	private function check_certificate_validity() {

		$cert_hash 			= sanitize_text_field( $_GET['cert_hash'] );
		$certificate_data 	= $this->course_certificate_data( $cert_hash );
		$course_id 			= $certificate_data[0]->course_id;
		$completion_date	= $certificate_data[0]->completion_date;

		$expiration 		= get_post_meta( $course_id, '_tutor_enable_certificate_expiration', true );
		$duration 			= get_post_meta( $course_id, '_tutor_certificate_expiration_duration', true );
		$duration_type 		= get_post_meta( $course_id, '_tutor_certificate_expiration_duration_type', true );


		if( 'on' === $expiration ) {
			$expire_date = strtotime( '+'. $duration .' '. $duration_type, strtotime( $completion_date ) );
			return $expire_date;
		}

		return false;
	}
	
	/**
	 * Get social share url
	 *
	 * @since v.1.0.0
	 */
	private function get_social_share_url( $platform ) {

		$cert_hash 			= sanitize_text_field( $_GET['cert_hash'] );
		$certificate_data 	= $this->course_certificate_data( $cert_hash );
		$description 		= 'Achievement';
		$image 		 		= $this->get_certificate_image( $certificate_data[0]->certificate_id, $cert_hash );
		
		switch( $platform ) {
			case 'facebook':
				return 'https://www.facebook.com/sharer.php?u='. $this->certificate_url();
				break;
			case 'twitter':
				return 'https://twitter.com/share?url='.$this->certificate_url().'&text=certificate&hashtags=achievement';
				break;
			case 'linkedin':
				return 'https://www.linkedin.com/sharing/share-offsite/?url='. $this->certificate_url();
				break;
			case 'pinterest':
				return 'https://www.pinterest.com/pin/create/button/?url='. $this->certificate_url().'&media='.$image.'&description='.$description;
				break;
			default:
				return $this->certificate_url();
				break;
		}
	}

	public function add_og_header_content() {
		
		if( isset( $_GET['cert_hash'] ) ){
			$cert_hash 	  = sanitize_text_field( $_GET['cert_hash'] );
			$data 		  = $this->course_certificate_data( $cert_hash );
			$course       = get_the_title( $data[0]->course_id );
			$title        = __( 'Course Completion Certificate', 'tutorlms-certificate-verify-and-share' );
			$description  = __( 'My course completion certificate for ', 'tutorlms-certificate-verify-and-share' ) . $course;
			$image 		  = $this->get_certificate_image( $data[0]->certificate_id, $cert_hash );
			
			echo '
			<meta property="og:title" content="' . $title . '"/>
			<meta property="og:description" content="' . $description . '"/>
			<meta property="og:image" content="' . $image . '"/>';
		}
	}

	/**
	 * Certificate image
	 *
	 * @since v.1.0.0
	 */
	private function get_certificate_image( $certificate_id, $cert_hash ) {
		
		$upload_dir 					= wp_upload_dir();
		$certificate_dir_url  			= $upload_dir['baseurl'] . '/' . 'tutor-certificates';
		$certificate_dir_path 			= $upload_dir['basedir'] . '/' . 'tutor-certificates';
		$rand_string 					= get_comment_meta( $certificate_id, 'tutor_certificate_has_image', true );
		$certificate_path 				= '/' . $rand_string . '-' . $cert_hash . '.jpg';
		$certificate_file 				= $certificate_dir_url . $certificate_path;

		$certificate_img = $certificate_dir_url . $certificate_path;
		return ! is_null( $certificate_img ) ? $certificate_img : 0;
	}

	/**
	 * Get completed course data
	 *
	 * @since v.1.0.0
	 */
	private function course_certificate_data( $cert_hash ) {
		global $wpdb;
		$data = $wpdb->get_results(
			"SELECT comment_ID as certificate_id,
					comment_post_ID as course_id,
					comment_author as completed_user_id,
					comment_date as completion_date,
					comment_content as completed_hash
			FROM	$wpdb->comments
			WHERE 	comment_agent = 'TutorLMSPlugin'
					AND comment_type = 'course_completed'
					AND comment_content = '$cert_hash';"
		);

		return ! empty( $data ) ? $data : [];
	}

	/**
	 * Register certificate verification shortcode
	 *
	 * @since v.1.0.0
	 */
	public function register_certificate_verification_shortcode() { 
		add_shortcode( 'certificate_verification', [ $this, 'load_certificate_verification_template'] );
	}

	public function load_certificate_verification_template() {

	 	if ( file_exists( plugin_dir_path( dirname( __FILE__ ) ) . 'public/partials/certificate-verification.php' ) ) {
			$file = plugin_dir_path( dirname( __FILE__ ) ) . 'public/partials/certificate-verification.php';
			
			ob_start(); // Start output buffering
			
			include $file;
			
			if( is_user_logged_in() ) {
				$text = sprintf( __( 'Certificate verification page loaded for user ID: %d', 'tutorlms-certificate-verify-and-share' ), get_current_user_id() );
			} else {
				$text = __( 'Certificate verification page loaded', 'tutorlms-certificate-verify-and-share' );
			}

			Tutorlms_Certificate_Verify_And_Share_WP_Logging::add( 
                $title          = __( 'Page load', 'tutorlms-certificate-verify-and-share' ), 
                $message        = $text,
                $post_parent    = 0, // ID of post where settings are loaded
                $log_type       = 'event' 
            );
			
			return ob_get_clean();
		}
	}

	public function add_expiry_date_on_certificate_page() {

		if( isset( $_GET[ 'cert_hash' ] ) ) {
			$cert_hash = sanitize_text_field( $_GET['cert_hash'] );
			wp_enqueue_script( $this->plugin_name.'expiry', plugin_dir_url( __FILE__ ) . 'js/tutorlms-certificate-expiry.js', array( 'jquery' ), $this->version, false );
		}
	}
	
	public function user_certificate_verification_callback() {

		if ( ! wp_verify_nonce( $_POST['nonce'], 'TutorLMS_Share_Certificate' ) ) {
    		wp_send_json( 'Request can not process right now.' );
		}
		
		if( isset( $_POST['URL'] ) ) {

    		$URL = $_POST[ 'URL' ];

			parse_str( parse_url( $URL, PHP_URL_QUERY ), $output );

			if( empty( $URL ) || empty( $output['cert_hash'] ) ) {
				echo '<h4 class="ttr-invld">' . esc_html( 'Invalid URL', 'tutorlms-certificate-verify-and-share' ) . '</h4>';
				die();
			}

			$data 			= $this->course_certificate_data( $output[ 'cert_hash' ] );
			$course_id 		= $data[0]->course_id;
            $expiration     = get_post_meta( $course_id, '_tutor_enable_certificate_expiration', true );
            $duration       = get_post_meta( $course_id, '_tutor_certificate_expiration_duration', true );
            $duration_type  = get_post_meta( $course_id, '_tutor_certificate_expiration_duration_type', true );
            $force_status   = get_comment_meta( $data[0]->certificate_id, '_certificate_status_'.$course_id, true );

			if( empty( $data ) ) {
				echo '<h4 class="ttr-invld">' . esc_html( 'Invalid URL' ) . '</h4>';
				die();
			}

			?>
		    <table cellspacing="0" cellpadding="10" style="width: 100%;" class="tlms-result-table">
        		<tbody>
            		<tr>
		                <td>Certificate Name</td>
		                <td>
		                    <span class="content"> <?php echo esc_html( get_the_title( absint( $data[0]->course_id ) ) ); ?></span>
		                    <span class="copy_content"> <i class="fas fa-copy"></i>
							<span class="dpshow">Copied</span>
						</span>
							
		                </td>
            		</tr>
            		<tr>
            		    <td>Issuing Organization</td>
            		    <td>
            		        <span class="content"> <?php echo esc_html( get_bloginfo( 'name' ) ); ?> </span>
            		        <span class="copy_content"> <i class="fas fa-copy"></i>
							<span class="dpshow">Copied</span>
						</span>
							
            		    </td>
            		</tr>
            		<tr>
            		    <td>Credential ID</td>
            		    <td>
            		        <span class="content"> <?php echo $output['cert_hash']; ?> </span>
            		        <span class="copy_content"> <i class="fas fa-copy"></i>
							<span class="dpshow">Copied</span>
						</span>
							
            		    </td>
            		</tr>
            		<tr>
            		    <td>Credential URL</td>
            		    <td>
            		        <span class="content"> <?php echo urldecode( $URL ); ?> </span>
            		        <span class="copy_content">
								<i class="fas fa-copy"></i>
								<span class="dpshow">Copied</span>
							</span>
							
            		    </td>
            		</tr>
            		<tr>
            		    <td>Issue Date</td>
            		    <td><?php echo esc_html( date('l jS \of F Y', strtotime( $data[0]->completion_date ) ) ); ?></td>
            		</tr>
            
        		<?php 
		        	if( 'on' === $expiration ) {
		        	    $expire_date = date( 'l jS \of F Y', strtotime( '+'. $duration .' '. $duration_type, strtotime( $data[0]->completion_date ) ) );
		        	?>
	                <tr>
	                    <td>Expire Date</td>
	                    <td><?php echo esc_html( $expire_date ); ?></td>
	                </tr>
            	<?php } ?>

            	<?php 
		        	if( 'active' !== $force_status || strtotime('now') >  strtotime( '+'. $duration .' '. $duration_type, strtotime( $data[0]->completion_date ) )  ) {
		        	?>
	                <tr class="tlms-cvs-invalid">
	                    <td><b><?php echo esc_html( 'Certificate Validation', 'tutorlms-certificate-verify-and-share' ); ?></b></td>
	                    <td><span><?php echo esc_html( 'Certificate is invalid', 'tutorlms-certificate-verify-and-share' ); ?></span></td>
	                </tr>
            	<?php } else { ?>
					<tr class="tlms-cvs-valid">
	                    <td><b><?php echo esc_html( 'Certificate Validation', 'tutorlms-certificate-verify-and-share' ); ?></b></td>
	                    <td><span><?php echo esc_html( 'Certificate is valid', 'tutorlms-certificate-verify-and-share' ); ?></span></td>
	                </tr>

            	<?php } ?>

        		</tbody>
    	</table>


			<?php
		}

		die();
	}

}	
