<?php

defined( 'ABSPATH' ) || exit;

class Play_Utils {

    protected static $_instance = null;

    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }

        return self::$_instance;
    }

    public function __construct() {
        add_action( 'wp', array( $this, 'setup_message' ) );

        function play_add_message( $message, $type = '' ) {
            return Play_Utils::instance()->add_message( $message, $type );
        }

        function play_get_template( $template_name, $args = array(), $template_path = '', $default_path = '' ) {
            return Play_Utils::instance()->get_template( $template_name, $args, $template_path, $default_path );
        }

        function play_get_template_html( $template_name, $args = array(), $template_path = '', $default_path = '' ) {
            return Play_Utils::instance()->get_template_html( $template_name, $args, $template_path, $default_path );
        }

        do_action( 'play_block_utils_init', $this );
    }

    /**
     * Validate the Nonce
     */
    public function validate_nonce( $nonce = '' ) {
        $nonce = isset( $_REQUEST[ 'nonce' ] ) ? $_REQUEST[ 'nonce' ] : $nonce;
        if ( ! wp_verify_nonce( sanitize_text_field( $nonce ), 'play_block_nonce' ) ) {
            return self::send_error();
        }

        return true;
    }

    /**
     * Send an Error Response
     *
     * @param $error string
     */
    public function send_error( $error = null ) {
        $error = ( $error ) ? $error : 'Invalid form field';

        return self::response( array(
            'status'  => 'error',
            'message' => $error
        ) );
    }

    /**
     * Send a response
     */
    public function response( $response ) {
        return wp_send_json( $response );
    }

    /**
     * Get template
     */
    public function get_template_html( $template_name, $args = array(), $template_path = '', $default_path = '' ) {
        ob_start();
        $this->get_template( $template_name, $args, $template_path, $default_path );

        return ob_get_clean();
    }

    /**
     * Get template html
     */
    public function get_template( $template_name, $args = array(), $template_path = '', $default_path = '' ) {
        $cache_key = sanitize_key( implode( '-', array( 'template', $template_name, $template_path, $default_path ) ) );
        $template  = (string) wp_cache_get( $cache_key, 'play' );

        if ( ! $template ) {
            $template = $this->locate_template( $template_name, $template_path, $default_path );
            wp_cache_set( $cache_key, $template, 'play' );
        }

        if ( ! empty( $args ) && is_array( $args ) ) {
            extract( $args );
        }

        do_action( 'play_before_template_part', $template_name, $template_path, $template, $args );
        include( $template );
        do_action( 'play_before_template_part', $template_name, $template_path, $template, $args );
    }

    public function locate_template( $template_name, $template_path = '', $default_path = '' ) {
        if ( ! $template_path ) {
            $template_path = '/templates/';
        }

        if ( ! $default_path ) {
            $default_path = plugin_dir_path( dirname( __FILE__ ) ) . '/templates/';
        }

        // Look within passed path within the theme - this is priority.
        $template = locate_template(
            array(
                trailingslashit( $template_path ) . $template_name,
                $template_name,
            )
        );

        // Get default template/.
        if ( ! $template ) {
            //$template = $default_path . $template_name;
            if ( 'php' != pathinfo( $template_name, PATHINFO_EXTENSION ) ) {
                $info = pathinfo( $template_name );
                $template_name = $info['filename'] . '.' . 'php';
            }
            
            $template = realpath( $default_path . $template_name );
        }

        // Return what we found.
        return apply_filters( 'play_locate_template', $template, $template_name, $template_path );
    }

    public function format_count( $number, $precision = 1 ) {
        if ( (int) $number == 0 ) {
            return '';
        }
        $precision = apply_filters( 'play_count_precision', $precision );

        $abbrevs = apply_filters( 'play_count_abbrevs', [ 12 => 'T', 9 => 'B', 6 => 'M', 3 => 'K', 0 => '' ] );

        foreach ( $abbrevs as $exponent => $abbrev ) {
            if ( abs( $number ) >= pow( 10, $exponent ) ) {
                $display  = $number / pow( 10, $exponent );
                $decimals = ( $exponent >= 3 && round( $display ) < 100 ) ? $precision : 0;
                $number   = number_format( $display, $decimals ) . $abbrev;
                break;
            }
        }

        return $number;
    }

    public function add_message( $message, $type = '' ) {
        // success is the default
        if ( empty( $type ) ) {
            $type = 'success';
        }

        // Send the values to the cookie for page reload display
        @setcookie( 'wp-message', $message, time() + 60 * 60 * 24, COOKIEPATH );
        @setcookie( 'wp-message-type', $type, time() + 60 * 60 * 24, COOKIEPATH );

    }


    public function setup_message() {
        if ( isset( $_COOKIE[ 'wp-message' ] ) ) {
            $template_message = stripslashes( $_COOKIE[ 'wp-message' ] );
        }

        if ( isset( $_COOKIE[ 'wp-message-type' ] ) ) {
            $template_message_type = stripslashes( $_COOKIE[ 'wp-message-type' ] );
        }

        add_action( 'template_notices', array( $this, 'render_message' ) );

        if ( isset( $_COOKIE[ 'wp-message' ] ) ) {
            @setcookie( 'wp-message', false, time() - 1000, COOKIEPATH );
        }
        if ( isset( $_COOKIE[ 'wp-message-type' ] ) ) {
            @setcookie( 'wp-message-type', false, time() - 1000, COOKIEPATH );
        }
    }

    public function render_message() {
        if ( isset( $_COOKIE[ 'wp-message' ] ) ) {
            $template_message = stripslashes( $_COOKIE[ 'wp-message' ] );

            if ( ! empty( $template_message ) ) :
                $type = ( 'success' == stripslashes( $_COOKIE[ 'wp-message-type' ] ) ) ? 'success' : 'error';
                $content      = apply_filters( 'render_message_content', stripslashes( $_COOKIE[ 'wp-message' ] ), $type ); ?>

                <div id="message" class="message template-message <?php echo $type; ?>">
                    <?php echo $content; ?>
                </div>

                <?php
                do_action( 'render_message' );
            endif;
        }
    }

    /**
     * Time duration
     *
     * @param $seconds int
     * @param $use     string
     * @param $simple  boolean
     */
    public function duration( $seconds, $use = null, $simple = false ) {
        if ( empty( $seconds ) || $seconds == 0 ) {
            return;
        }
        $periods = array(
            'years'   => 31556926,
            'Months'  => 2629743,
            'weeks'   => 604800,
            'days'    => 86400,
            'hours'   => 3600,
            'minutes' => 60,
            'seconds' => 1
        );

        $seconds  = (float) $seconds;
        $segments = array();
        foreach ( $periods as $period => $value ) {
            if ( $use && strpos( $use, $period[ 0 ] ) === false ) {
                continue;
            }
            $count = floor( $seconds / $value );
            if ( $count == 0 && ! $simple ) {
                continue;
            }
            if ( $count == 0 && $simple && $period !== 'minutes' ) {
                continue;
            }
            $segments[ strtolower( $period ) ] = $count;
            $seconds                           = $seconds % $value;
        }

        $string = array();
        foreach ( $segments as $key => $value ) {
            $segment_name = substr( $key, 0, - 1 );
            $segment      = $simple ? $value : ( $value . ' ' . $segment_name );
            if ( $value != 1 && ! $simple ) {
                $segment .= 's';
            }
            if ( $value < 10 && $simple ) {
                $segment = '0' . $segment;
            }
            $string[] = $segment;
        }

        $string = apply_filters( 'play_duration', $string, $seconds, $use, $simple );

        return $simple ? implode( ':', $string ) : implode( ', ', $string );
    }

    public function timeToMS( $time ) {
        $sec = 0;
        foreach ( array_reverse( explode(':', $time) ) as $k => $v ){
            $sec += pow( 60, (int)$k ) * (int)$v;
        }
        return $sec * 1000;
    }

    public function fixURL( $url ) {
        $site_url = site_url();
        $site_uri = preg_replace('#^https?://#i', '', $site_url);
        if( strpos( $url, $site_uri ) !== false ){
            if(strpos($site_url, 'https://') !== false){
                $url = str_replace('http://', 'https://', $url);
            }else{
                $url = str_replace('https://', 'http://', $url);
            }
            return $url;
        }else{
            return false;
        }
    }
}

Play_Utils::instance();
